#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * I/O utilities for client server
 *
 * Should be rethought, cleaned up, and moved to libsos
 *
 * Well, it was moved to libsos, but it wasn't really rethunk and cleaned.
 */

#include "sos.h"

#ifndef US_EXPORT_VERSION
#endif /* ! US_EXPORT_VERSION */


/*
 * Static I/O timout value
 */
static int sos_io_timeout = 0;
static struct sigaction old_sigaction;	/* Caches old signal vec. */

static int sos_io_with_timeout(int fd, char *addr, int len, int (*ior) (int, caddr_t, __SIZE_TYPE__));


/*
 * This function exists for no purpose but to prevent the program from aborting
 * on a SIGALRM (default behavior).
 */
static void
sos_ioTimeout(void)
{
  /* Restore old signal behavior */
  sigaction(SIGALRM, (struct sigaction *)&old_sigaction,
	    (struct sigaction *)NULL);
  return;
}

static struct sigaction iotimeout = SIGACTIONDEFINE( sos_ioTimeout, 0, _INTERRUPT_SYSCALLS_ );



/*
 * Read the data raw from the net.  Possibly decrypt.
 * non-encrypted is short-cut immediately
 */
int
sosRawRead(int fd, caddr_t addr, __SIZE_TYPE__ len)
{
  SOS_ENTRY("sos_util","sosRawRead",NULL);
#ifndef US_EXPORT_VERSION
#endif /* ! US_EXPORT_VERSION */
      SOS_ERETURN (sos_io_with_timeout(fd, addr, len, (int (*)(int, caddr_t, __SIZE_TYPE__))read));
#ifndef US_EXPORT_VERSION
#endif /* ! US_EXPORT_VERSION */
}



/****************************************************************************
 **
 ** sosRawWrite
 **
 ** Perform a write with Brimstone options:
 **
 ** io timeouts
 ** encryption
 **
 ** We are encrypting in a stream cipher, so we don't need to worry
 ** about the encryption block-size.  However, we are encrypting the
 ** data in the user's supplied buffer, so the user must NOT change
 ** the data in the event of a partial write, and must NOT look at
 ** the data in any event.
 **
 ** XXX - need to be able to control options (DES, timeouts) on a
 ** per descriptor basis.
 ***************************************************************************/
int
sosRawWrite(int fd, caddr_t addr, __SIZE_TYPE__ len)
{
  SOS_ENTRY("sos_util","sosRawWrite",NULL);
#ifndef US_EXPORT_VERSION
#endif /* ! US_EXPORT_VERSION */
      SOS_ERETURN(sos_io_with_timeout(fd, addr, len, (int (*)(int, caddr_t, __SIZE_TYPE__))write));
#ifndef US_EXPORT_VERSION
#endif /* ! US_EXPORT_VERSION */
}



/*
 * Generic I/O with timeout.  Interestingly enough, we don't need to
 * know whether this is a read or a write :-)
 *
 * It is not necessarily a good idea to have our own private alarms
 */
static int
sos_io_with_timeout(int fd, char *addr, int len, int (*ior) (int fd, caddr_t addr, __SIZE_TYPE__ len))
{
  SOS_ENTRY("sos_util","sos_io_with_timeout",NULL);
  int save_errno;
  int nbytes;
  struct itimerval newit,oldit;

  if (sos_io_timeout)
    {
      sigaction(SIGALRM, (struct sigaction *)&iotimeout,
		(struct sigaction *)&old_sigaction);
      newit.it_interval.tv_sec = 0;
      newit.it_interval.tv_usec = 0;
      newit.it_value.tv_sec = sos_io_timeout;
      newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, &oldit);
    }

  nbytes = (*ior) (fd, addr, len);

#if defined SOS_IO_DEBUG
  /* Use with caution! At least don't have encryption turned on. */
  {
    int i;
    char *p;

    printf("sos_io_with_timeout(): performed I/0: addr: *-*");
    for (p = addr, i = 0; i < len; i++, p++)
      printf("%c", *p);
    printf("*-*\n");
  }
#endif	/* SOS_IO_DEBUG */

  save_errno = errno;
  if (sos_io_timeout)
    {
      setitimer(ITIMER_REAL, &oldit, NULL);
      sigaction(SIGALRM, (struct sigaction *)&old_sigaction,
		(struct sigaction *)NULL);
    }

  errno = save_errno;

  if (nbytes < 0 && errno != EWOULDBLOCK && errno != EAGAIN)
    {
      sos_error_printf("I/O function failed: %s\n",strerror(errno));
    }

  errno = save_errno;
  SOS_ERETURN(nbytes);
}



/*
 * Set the current timeout interval
 */
void
sos_set_io_timeout(int timeout)
{
  sos_io_timeout = timeout;
}



/*
 * Initialize encryption on both sides
 *
 * fd: socket to send initial message. 
 */
int
sos_encrypt_init(char *key,int fd)
{
  SOS_ENTRY("sos_util","sos_encrypt_init",NULL);
#ifndef US_EXPORT_VERSION
#else /* ! US_EXPORT_VERSION */
  SOS_RETURN (0);
#endif /* ! US_EXPORT_VERSION */
}



/*
 * Call sosRawRead while making sure we read the number of bytes we wanted to read
 */
int
sosBIOread(int fd, caddr_t addr, __SIZE_TYPE__ len)
{
  SOS_ENTRY("sos_util","sosBIOread",NULL);
  int width = fd + 1;
  fd_set fds;
  int nbytes = 0;
  int curlen = len;
  int save_errno;


  FD_ZERO(&fds);
  FD_SET(fd, &fds);		/* If FD is invalid, this can corrupt memory and/or coredump */

  while (curlen > 0)
    {
      if (select(width, &fds, NULL, NULL, NULL) < 0)
	{
	  save_errno = errno;
	  sos_error_printf("Select failed: %s\n",strerror(errno));
	  errno = save_errno;
	  SOS_ERETURN(-1);
	}

      if ((nbytes = sosRawRead(fd, addr, curlen)) < 0)
	{
	  save_errno = errno;
	  sos_error_printf("Could not read after select said was OK\n");
	  errno = save_errno;
	  SOS_ERETURN(-1);
	}

      if (nbytes == 0)
	SOS_RETURN(0);

      curlen -= nbytes;
      addr += nbytes;
    }

  SOS_RETURN(len);
}



/*
 * Call sosRawWrite while making sure we write the number of bytes we wanted to write
 */
int
sosBIOwrite(int fd, caddr_t addr, __SIZE_TYPE__ len)
{
  SOS_ENTRY("sos_util","sosBIOwrite",NULL);
  int width = fd + 1;
  fd_set fds;
  int curlen = len;
  int nbytes = 0;
  int save_errno;


  FD_ZERO(&fds);
  FD_SET(fd, &fds);		/* If FD is invalid, this can corrupt memory and/or coredump */

  while (curlen > 0)
    {
      if (select(width, NULL, &fds, NULL, NULL) < 0)
	{
	  save_errno = errno;
	  sos_error_printf("Select failed: %s\n",strerror(errno));
	  errno = save_errno;
	  SOS_ERETURN(-1);
	}

      if ((nbytes = sosRawWrite(fd, addr, curlen)) < 0)
	{
	  save_errno = errno;
	  sos_error_printf("Write failed after select said was OK\n");
	  errno = save_errno;
	  SOS_ERETURN(-1);
	}

      if (nbytes == 0)
	SOS_RETURN(0);

      curlen -= nbytes;
      addr += nbytes;
    }

  SOS_RETURN(len);
}



/*
 * Call sosRawRead then rip off the CR/LF stuff
 *
 * DEPRECATED
 */
int
sosRead(int fd, caddr_t addr, __SIZE_TYPE__ len)
{
  SOS_ENTRY("sos_util","sosRead",NULL);
  int nbytes = sosRawRead(fd, addr, len);
  int save_errno = errno;

  if (nbytes <= 0)
    {
      if (nbytes < 0) sos_error_printf("invalid length\n");
      errno = save_errno;
      SOS_ERETURN(nbytes);
    }

  /*
   * Null out first CR/LF.  Note this means and data on the 2nd - nth
   * lines has just been lost.  Ach, poor lassie.
   */
  sos_rip(addr);

  SOS_RETURN(nbytes);
}



/*
 * Call sosRawWrite then do nothing
 *
 * DEPRECATED
 */
int
sosWrite(int fd, caddr_t addr, __SIZE_TYPE__ len)
{
  SOS_ENTRY("sos_util","sosWrite",NULL);
  int nbytes;
  int save_errno;

  /*
   * For compatibility. Sigh...
   */
  if ((nbytes = sosRawWrite(fd, addr, len)) < 0)
    {
      save_errno = errno;
      sos_error_printf("sosRawWrite was not happy\n");
      errno = save_errno;
      SOS_ERETURN(-1);
    }

  SOS_RETURN(nbytes);
}



/*
 * Initial encryption fd flags
 */
void
sos_encrypt_on_init_fun(void)
{
  SOS_ENTRY("sos_util","sos_encrypt_on_init_fun",NULL);
#ifndef US_EXPORT_VERSION
#endif /* ! US_EXPORT_VERSION */

  SOS_VRETURN();
}


/*
 * Count the columns taken up by an integer
 */
int sos_intcols(int x)
{
  SOS_ENTRY("sos_util","sos_intcols",NULL);
  int negsign=0;
  int ret=1;
  
  if ( x < 0 )
    {
      negsign++;
      x=-x;
    }

  if ( x > 0 )
    ret=((int)(log10((double)x)))+1+negsign;

  SOS_RETURN(ret);
}
