/*
 * Author: Paul.Russell@rustcorp.com.au
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <netinet/in.h>
#include <net/ethernet.h>
#include <string.h>

#include <libipfwc.h>
#include "fwinterface.h"

#ifndef MIN
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif

#define IP_FIREWALL_MONITOR_DEVICE "/dev/fwmonitor"
#define IP_FIREWALL_ETHERTAP_DEVICE "/dev/tap0"

static int fw_handle = -1;
static __u32 fw_rulenum = 0;
static int fw_reinject_handle = -1;
static int fw_raw_socket = -1;

/* Simple linked list: may make this more sophisticated later */
struct fw_monitor_info
{
    struct ip_fw fw;
    ip_chainlabel policy;
    ip_chainlabel label;
    fw_readpktfn *fn;
    int tap_inject;

    struct fw_monitor_info *next;
};

static struct fw_monitor_info *fw_info = NULL;

static struct fw_monitor_info *find_info(__u32 rulenum)
{
    struct fw_monitor_info *i;
    for (i = fw_info; i; i = i->next) {
	if (i->fw.fw_mark == rulenum) return i;
    }
    return NULL;
}

static struct fw_monitor_info *new_info(void)
{
    struct fw_monitor_info *info = malloc(sizeof(struct fw_monitor_info));
    if (!info) {
	fprintf(stderr, "Out of memory.\n");
	exit(1);
    }

    info->next = fw_info;
    fw_info = info;

    return fw_info;
}

static void delete_info(struct fw_monitor_info *info)
{
    struct fw_monitor_info **infoptr = &fw_info;
    struct fw_monitor_info *tmp;

    while (*infoptr != info) {
	assert(*infoptr);
	*infoptr = (*infoptr)->next;
    }

    tmp = *infoptr;
    *infoptr = tmp->next;
    free(tmp);
}

/* Manually insert/append a rule */
static int fw_add_rule(const struct ip_fw *packetspecs, 
		       const ip_chainlabel chainname,
		       const char *policy,
		       int position,
		       int silent)
{
    struct ip_fwuser fw;
    int ret;

    fw.ipfw = *packetspecs;
    strcpy(fw.label, policy);

    if (position == -1) {
	/* Append */
	ret = ipfwc_append_entry(chainname, &fw);
    }
    else {
	/* Insert */
        ret = ipfwc_insert_entry(chainname, &fw, position);
    }

    if (!ret && !silent) 
	fprintf(stderr, "adding rule: %s\n", ipfwc_strerror(errno));
    return ret;
}

/* Manually delete a rule */
static int fw_delete_rule(const struct ip_fw *packetspecs, 
			  const ip_chainlabel chainname,
			  const char *policy,
			  int silent)
{
    struct ip_fwuser fw;
    int ret;

    fw.ipfw = *packetspecs;
    strcpy(fw.label, policy);

    ret = ipfwc_delete_entry(chainname, &fw);
    if (!ret && !silent)
	fprintf(stderr, "delete of rule failed: %s\n", ipfwc_strerror(errno));

    return ret;
}

/* Must be called before any other routines. */
int fw_init_listen(int need_injection, int silent)
{
    fw_handle = open(IP_FIREWALL_MONITOR_DEVICE, O_RDONLY);
    if (fw_handle < 0) {
	if (!silent) perror("opening " IP_FIREWALL_MONITOR_DEVICE);
	return 0;
    }
    if (need_injection) {
	fw_reinject_handle = open(IP_FIREWALL_ETHERTAP_DEVICE, O_RDWR);
	if (fw_reinject_handle == -1) {
	    if (!silent) perror("opening " IP_FIREWALL_ETHERTAP_DEVICE);
	    return 0;
	}
    }

    fw_raw_socket = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
    fprintf(stderr, "fw_raw_socket = %i.\n", fw_raw_socket);
    if (fw_raw_socket == -1) {
	if (!silent) perror("creating raw socket");
	return 0;
    }

#if 0
    if (setsockopt(fw_raw_socket,IPPROTO_IP,IP_HDRINCL,&off,sizeof(off)) < 0) {
	perror("Setting no IP_HDRINCL");
	return 0;
    }
#endif

    return 1;
}

/* Does a cleanup; call before terminating. */
void fw_close_listen(int silent)
{
    while (fw_info) {
	fw_deregister_interest(fw_info->fw.fw_mark, 
			       silent);
    }
    if (fw_handle >= 0) close(fw_handle);
    if (fw_reinject_handle >= 0) close(fw_reinject_handle);
}

/* Register interest in a certain type of packet; returns 0 for failure,
   or a unique rule number. */
__u32 fw_register_interest(const struct ip_fw *packetspecs, 
			   const ip_chainlabel chainname,
			   fw_readpktfn *fn,
			   const char *policy,
			   int is_input,
			   int silent)
{
    struct fw_monitor_info *info = new_info();

    assert(!policy || strlen(policy) < sizeof(ip_chainlabel));

    memcpy(&info->fw, packetspecs, sizeof(info->fw));

    if (policy) strcpy(info->policy, policy);
    else info->policy[0] = '\0';

    info->tap_inject = is_input;
    info->fw.fw_mark = (((__u32)getpid())<<16) |++fw_rulenum;
    info->fw.fw_flg |= (IP_FW_F_MARKABS | IP_FW_F_NETLINK);
    strcpy(info->label, chainname);
    info->fn = fn;

    fprintf(stderr, "mark = %u\n", info->fw.fw_mark);

    if (fw_add_rule(&info->fw, info->label, info->policy, 1, silent))
	return info->fw.fw_mark;
    else return 0;
}

/* Deregister interest; returns 0 for failure. */
int fw_deregister_interest(__u32 rulenum, int silent)
{
    /* Rule deletion-by-match doesn't cover mark values.  This may
     * eventually change, but until then there is the risk of
     * deleting another monitor's rule if it is identical (but for
     * the mask). */
    struct fw_monitor_info *info = find_info(rulenum);
    int ret;

    if (!info) {
	if (!silent) 
	    fprintf(stderr, "fw_unregister_interest on `%u' failed: no such rule registered", rulenum);
	return 0;
    }

    ret = fw_delete_rule(&info->fw, info->label, info->policy, silent);
    delete_info(info);

    return ret;
}

/* Use one of the two interfaces below.  Either simply call
 * "fw_do_listen();" from main(), or for more complex programs,
 * use fw_get_handle(), then fw_do_event(). */

/* Simple interface: runs until something goes wrong. */
int fw_do_listen(int silent)
{
    fd_set readset;

    /* Did they call fw_init_listen()? */
    assert(fw_handle >= 0);

    for(;;) {
	FD_ZERO(&readset);
	FD_SET(fw_handle, &readset);

	if (select(fw_handle+1, &readset, NULL, NULL, NULL) != 1) {
	    if (errno != EINTR) {
		if (!silent) perror("doing select");
		return 0;
	    }
	}
	else if (!fw_do_event(silent)) return 0;
    }
}
	
/* More complex interface. */
/* Returns -1 or a filehandle */
int fw_get_handle()
{
    return fw_handle;
}

struct netdev {
    __u32 len;
    __u32 mark;
    char interface[IFNAMSIZ];
};

/* Returns 0 if there's a problem, otherwise handles a read */
int fw_do_event(int silent)
{
    /* One nice structure for reading from netdev, and writing to
     * ethertap device.  IFNAMSIZ is a multiple of 4 and 8 (yay!).
     * Look Mum, no casts!
     */
    struct {
	union {
	    struct {
		__u8 prepad[sizeof(struct netdev) - 16];
		__u8 hdr[16];
	    } eth;
	    struct netdev netdev;
	} hdr;
	union {
	    __u8 typeless[65535];
	    struct iphdr iphdr;
	} pkt;
    } data;
    struct fw_monitor_info *info;
    int bytes;

#if 0
    /* This doesn't work; discards entire netlink packet. */
    if (read(fw_handle, &data.hdr.netdev.len, sizeof(__u32)) != sizeof(__u32)){
	if (!silent) perror("doing initial read");
	return 0;
    }
    assert(data.hdr.netdev.len >= sizeof(data.hdr.netdev));
    bytes = read(fw_handle, &data, data.hdr.netdev.len);
#else
    /* Test for bogus read (2.1.125).  Sometimes it doesn't actually fill
     * in any data.  This is a temporary error caused by overloading. */
    data.hdr.netdev.len = 0;
    bytes = read(fw_handle, &data, sizeof(data));
    if (data.hdr.netdev.len == 0) {
	    return 1;
    }
#endif

    /* Assuming atomic read here */
    if (bytes != data.hdr.netdev.len) {
	if (!silent) {
	    if (bytes >= 0) 
		fprintf(stderr, "doing read: got %i bytes not %u\n",
			bytes, data.hdr.netdev.len);
	    else
		perror("doing read");
	}
	return 0;
    }

    if ((info = find_info(data.hdr.netdev.mark)) != NULL) {
	const struct iphdr *newpkt;

	/* Should never get partial packets */
	assert(bytes == (MIN(info->fw.fw_outputsize,
			    ntohs(data.pkt.iphdr.tot_len))
			 + sizeof(data.hdr.netdev)));

	newpkt = info->fn(data.hdr.netdev.mark, data.hdr.netdev.interface, 
			  info->label, 
			  bytes - sizeof(data.hdr.netdev), 
			  &data.pkt.iphdr);
	if (newpkt) {
	    if (newpkt != &data.pkt.iphdr) {
		fprintf(stderr, "New packet.\n");
		/* They handed us a new packet.  Copy over */
		memmove(&data.pkt.iphdr, newpkt, ntohs(newpkt->tot_len));
	    }

	    if (info->tap_inject) {
		/* Ethertap */
		memset(data.hdr.eth.hdr, 0, sizeof(data.hdr.eth.hdr));

		/* Ethernet header is 14 bytes, but kernel aligns it by
		 * prepadding 2 bytes. */
		*((__u16 *)(data.hdr.eth.hdr+14)) = htons(ETH_P_IP);

		if (write(fw_reinject_handle, data.hdr.eth.hdr, 
			  sizeof(data.hdr.eth.hdr) + ntohs(data.pkt.iphdr.tot_len))
		    != sizeof(data.hdr.eth.hdr) + ntohs(data.pkt.iphdr.tot_len)) {
		    if (!silent) perror("sending raw packet through ethertap");
		    return 0;
		}
	    }
	    else {
		struct sockaddr_in dst;

		dst.sin_family = AF_INET;
		dst.sin_addr.s_addr = data.pkt.iphdr.daddr;
		/* Raw socket. */
		{
			unsigned int i;
			for (i = 0; i < 40; i++)
				printf("0x%02x ",((unsigned char *)&data.pkt.iphdr)[i]);
			printf("\n");
			for (i = 0; i < sizeof(dst); i++)
				printf("0x%02x ", ((unsigned char *)&dst)[i]);
			printf("\n");
		}
		if (sendto(fw_raw_socket, &data.pkt.iphdr, 
			   ntohs(data.pkt.iphdr.tot_len),
			   0, (struct sockaddr *)&dst, sizeof(dst)) < 0) {
		    if (!silent) 
			perror("sending raw packet through socket");
		    return 0;
		}
	    }
	}
    }
    return 1;
}

