/********************************************************************
*                                                                   *
* curvesExample.c - Curves Plot Widget Example Program              *
*                                                                   *
* This program is an example program that demonstrates the Curves   *
* widget capabilities                                               *
*                                                                   *
*                                                                   *
* July 15, 1994 by George Dimas                                     *
*                                                                   *
********************************************************************/
#include <stdlib.h>
#include <limits.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>
#include <Xm/PushB.h>
#include <Xm/MessageB.h>
#include <Xm/DialogS.h>
#include <Xm/Text.h>
#include <Xm/LabelG.h>
#include "../plot_widgets/Curves.h"
#include "../util/printUtils.h"

#define NUM_POINTS  10
#define NUM_CURVES  3

Widget CurveWidget, TimeSeriesToggle, ErrorBarsToggle;
Widget RangeText[4];

static void createButton(Widget parent, char *name, char *label,
	void *callbackProc, XtPointer cbArg);
static Widget createToggleButton(Widget parent, char *name, char *label,
	Boolean initialState, void *callbackProc, XtPointer cbArg);
static void logXCB(Widget w, caddr_t client_data, caddr_t call_data);
static void logYCB(Widget w, caddr_t client_data, caddr_t call_data);
static void doubleBufferCB(Widget w, caddr_t client_d, caddr_t call_d);
static void setRangeCB(Widget w, caddr_t client_data, caddr_t call_data);
static void rangeOkCB(Widget w, Widget x, caddr_t call_data);
static void rangeCancelCB(Widget w, Widget x, caddr_t call_data);
static void createWarningMessage(Widget parent, char *text);
static void getRangeCB(Widget w, caddr_t client_data, caddr_t call_data);
static void scaleVariablesCB(Widget w, caddr_t client_data, caddr_t call_data);
static void setMarkLineStyleCB(Widget w, caddr_t client_d, caddr_t call_data);
static void showLegendCB(Widget w, caddr_t client_d, caddr_t call_d);
static void timeSeriesCB(Widget w, XtPointer client_d, caddr_t call_d);
static void errorBarsCB(Widget w, XtPointer client_d, caddr_t call_d);
static void createCurves(Boolean isTimeSeries, Boolean showErrorBars);
static void zoomCB(Widget w, XtPointer data, caddr_t call_data);
static void printCB(Widget w, caddr_t client_d, caddr_t call_d);
static void exitCB(Widget w, caddr_t client_data, caddr_t call_data);

main(int argc, char *argv[])
{
    XtAppContext appContext;
    Widget toplevel, form, checkBox, rowColumn;
    XmString s1, s2;
    
    /* Initialize X and Xt */
    toplevel = XtAppInitialize(&appContext, "CurvesTest", NULL,
	    0, &argc, argv, NULL, NULL, 0);
    
    /* Create a form Widget as the container for a row column,
       check box and curve widget. */
    form = XtCreateManagedWidget("form",
    	xmFormWidgetClass, toplevel, NULL, 0);
    
    /* Create a curve widget. */
    CurveWidget = XtVaCreateManagedWidget("CurveWidget",
    	    curvesWidgetClass, form, 
    	    XmNxAxisLabel, s1=XmStringCreateSimple("X axis label"),
    	    XmNyAxisLabel, s2=XmStringCreateSimple("This is the Y axis label"),
    	    XmNtopAttachment, XmATTACH_FORM,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, 21,
    	    XmNrightAttachment, XmATTACH_FORM,
    	    XmNbottomAttachment, XmATTACH_FORM,
    	    NULL);
    XmStringFree(s1);
    XmStringFree(s2);

    /* Draw the example curves */
    createCurves(TRUE, TRUE);
    	
    /* Create a row column to hold toggle buttons. */
    checkBox = XtVaCreateManagedWidget("toggleButtons",
    	    xmRowColumnWidgetClass, form,
    	    XmNtopAttachment, XmATTACH_FORM,
    	    XmNleftAttachment, XmATTACH_FORM,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, 20,
    	    NULL);
    
    /* Create the toggle buttons */
    createToggleButton(checkBox, "logX", "Log X", FALSE, logXCB, NULL);
    createToggleButton(checkBox, "logY", "Log Y", FALSE, logYCB, NULL);
    createToggleButton(checkBox, "showLegend",
    	    "Show Legend", TRUE, showLegendCB, NULL);
    TimeSeriesToggle = createToggleButton(checkBox, "timeSeries",
    	    "Time Series", TRUE, timeSeriesCB, NULL);
    ErrorBarsToggle = createToggleButton(checkBox, "showErrorBars",
    	    "Show Error Bars", TRUE, errorBarsCB, NULL);
    createToggleButton(checkBox, "doubleBuffer",
    	    "Double Buffer", FALSE, doubleBufferCB, NULL);
    
    /* Create a row column to hold the push buttons. */
    rowColumn = XtVaCreateManagedWidget("rowColumn",
    	    xmRowColumnWidgetClass, form,
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, checkBox,
    	    XmNtopOffset, 10,
    	    XmNleftAttachment, XmATTACH_FORM,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, 20,
    	    XmNbottomAttachment, XmATTACH_FORM,
    	    NULL);
    
    /* Create the push buttons. */
    createButton(rowColumn, "setMarkLineStyle", "Set Mark & Line Style...",
	    setMarkLineStyleCB, NULL);
    createButton(rowColumn, "scaleVariables", "Scale Variables...",
	    scaleVariablesCB, NULL);
    createButton(rowColumn, "setRange", "Set Visible Range...",
	    setRangeCB, NULL);
    createButton(rowColumn, "getRange", "Get Visible Range...",
	    getRangeCB, NULL);
    createButton(rowColumn, "zoomIn", "Zoom In", zoomCB, (XtPointer)1);
    createButton(rowColumn, "zoomOut", "Zoom Out", zoomCB, (XtPointer)2);
    createButton(rowColumn, "resetZoom", "Reset Zoom",
    	    zoomCB, (XtPointer)3);
    createButton(rowColumn, "printPlot", "Print Plot...",
	    printCB, NULL);
    createButton(rowColumn, "exitProgram", "Exit Program",
	    exitCB, NULL);
    
    /* Realize widgets and call XtMainLoop to continuously process events. */
    XtRealizeWidget(toplevel);
    XtAppMainLoop(appContext);
}

static void createButton(Widget parent, char *name, char *label,
	void *callbackProc, XtPointer cbArg)
{
    XmString s1;
    Widget button;
    	   	  	
    button = XtVaCreateManagedWidget(name, xmPushButtonWidgetClass, parent,
    	    XmNlabelString, s1=XmStringCreateSimple(label), NULL);
    XtAddCallback(button, XmNactivateCallback,
    	    (XtCallbackProc)callbackProc, cbArg); 
    XmStringFree(s1);
}

static Widget createToggleButton(Widget parent, char *name, char *label,
	Boolean initialState, void *callbackProc, XtPointer cbArg)
{
    XmString s1;
    Widget button;
    	   	  	
    button = XtVaCreateManagedWidget(name, xmToggleButtonWidgetClass, parent,
    	    XmNlabelString, s1=XmStringCreateSimple(label),
    	    XmNset, initialState, NULL);
    XtAddCallback(button, XmNvalueChangedCallback,
    	    (XtCallbackProc)callbackProc, cbArg); 
    XmStringFree(s1);
    return button;
}

static void logXCB(Widget w, caddr_t client_data, caddr_t call_data)
{ 
    /* This routine is activated every time toggle button Log X has
       been pressed. It changes the X axis from linear to log scaling
       and vice versa. */
    
    XtVaSetValues(CurveWidget, XmNxLogScaling, XmToggleButtonGetState(w), NULL);
}

static void logYCB(Widget w, caddr_t client_data, caddr_t call_data)
{
    
    /* This routine is activated every time toggle button Log Y has
       been pressed. It changes the Y axis from linear to log scaling
       and vice versa. */
    
    XtVaSetValues(CurveWidget, XmNyLogScaling, XmToggleButtonGetState(w), NULL);
}

static void doubleBufferCB(Widget w, caddr_t client_d, caddr_t call_d)
{  
    /* This routine is activated every time toggle button Double Buffer 
       has been pressed. It turns on and off graphics buffering */
    
    XtVaSetValues(CurveWidget, XmNdoubleBuffer, XmToggleButtonGetState(w),
    	    NULL);
}

static void setRangeCB(Widget w, caddr_t client_data, caddr_t call_data)
{
    /* This function will create and pop-up a form dialog, in which a user
       can enter the maximum and minimum values for the new range.
       Also, included is a Cancel and an Ok push button. The callback for
       the Ok pushbutton actually sets the visible range */ 
       
    Widget form, labelMinX, labelMinY, labelMaxX, labelMaxY, ok, cancel;
    XmString xmstr; 
    
    /* Create a form widget to contain the text & label widgets. */
    form = XmCreateFormDialog(w, "form", NULL, 0);
    XtVaSetValues(form, 
    	    XmNdialogTitle, xmstr=XmStringCreateSimple("Set Range"),
    	    XmNautoUnmanage, FALSE, 
    	    NULL);
    XmStringFree(xmstr); 
    

    /* Create 4 label & 4 text widgets. */
    labelMinX = XtVaCreateManagedWidget("labelMinX", 
    	    xmLabelGadgetClass, form,
    	    XmNlabelString, xmstr=XmStringCreateSimple("min X"),
    	    XmNtopAttachment, XmATTACH_FORM,
    	    XmNtopOffset, 10,
    	    XmNleftAttachment, XmATTACH_FORM,
    	    XmNleftOffset, 10,
    	    NULL);
    XmStringFree(xmstr); 
    
    RangeText[0] = XtVaCreateManagedWidget("textMinX", 
    	    xmTextWidgetClass, form,
    	    XmNtopAttachment, XmATTACH_FORM,
    	    XmNtopOffset, 5,
    	    XmNleftAttachment, XmATTACH_WIDGET,
    	    XmNleftWidget, labelMinX,
    	    NULL); 
    	
    labelMinY = XtVaCreateManagedWidget("labelMinY", 
    	    xmLabelGadgetClass, form,
    	    XmNlabelString, xmstr=XmStringCreateSimple("min Y"),
    	    XmNtopAttachment, XmATTACH_FORM,
    	    XmNtopOffset, 10,
    	    XmNleftAttachment, XmATTACH_WIDGET,
    	    XmNleftWidget, RangeText[0],
    	    XmNleftOffset, 10,
    	    NULL);
    XmStringFree(xmstr);
    	
    RangeText[1] = XtVaCreateManagedWidget("textMinY", 
    	    xmTextWidgetClass, form,
    	    XmNtopAttachment, XmATTACH_FORM,
    	    XmNtopOffset, 5,
    	    XmNleftAttachment, XmATTACH_WIDGET,
    	    XmNleftWidget, labelMinY,
    	    NULL);
    	
    labelMaxX = XtVaCreateManagedWidget("labelMaxX", 
    	    xmLabelGadgetClass, form,
    	    XmNlabelString, xmstr=XmStringCreateSimple("max X"),
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, labelMinX,
    	    XmNtopOffset, 20,
    	    XmNleftAttachment, XmATTACH_FORM,
    	    XmNleftOffset, 10,
    	    NULL);
    XmStringFree(xmstr);
    
    RangeText[2] = XtVaCreateManagedWidget("textMaxX", 
    	    xmTextWidgetClass, form,
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, RangeText[0],
    	    XmNtopOffset, 5,
    	    XmNleftAttachment, XmATTACH_WIDGET,
    	    XmNleftWidget, labelMaxX,
    	    NULL); 
    	
    labelMaxY = XtVaCreateManagedWidget("labelMaxY", 
            xmLabelGadgetClass, form,
            XmNlabelString, xmstr=XmStringCreateSimple("max Y"),
            XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, labelMinY,
    	    XmNtopOffset, 20,
    	    XmNleftAttachment, XmATTACH_WIDGET,
    	    XmNleftWidget, RangeText[2],
    	    XmNleftOffset, 10,
    	    NULL);
    XmStringFree(xmstr); 
    
    RangeText[3] = XtVaCreateManagedWidget("textMaxY", 
    	    xmTextWidgetClass, form,
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, RangeText[1],
    	    XmNtopOffset, 5, 
    	    XmNleftAttachment, XmATTACH_WIDGET,
    	    XmNleftWidget, labelMaxY,
    	    NULL);	

    	
    /* Create OK push button widget and callback */
    ok = XtVaCreateManagedWidget("ok", 
    	    xmPushButtonWidgetClass, form,
    	    XmNlabelString, xmstr=XmStringCreateSimple(" OK "),
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, RangeText[2],
    	    XmNtopOffset, 20,
    	    XmNleftAttachment, XmATTACH_FORM,
    	    XmNleftOffset, 20,  
    	    NULL);
    XtAddCallback(ok, XmNactivateCallback, 
    	    (XtCallbackProc) rangeOkCB, (void *) form);
    XmStringFree(xmstr);
    
    /* Create Cancel push button widget and callback */ 
    cancel = XtVaCreateManagedWidget("cancel", 
    	    xmPushButtonWidgetClass, form,
    	    XmNlabelString, xmstr=XmStringCreateSimple("Cancel"),
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, RangeText[2],
    	    XmNtopOffset, 20,
    	    XmNleftAttachment, XmATTACH_WIDGET,
    	    XmNleftWidget, ok,
    	    XmNleftOffset, 5, 
    	    NULL);
    XtAddCallback(cancel, XmNactivateCallback, 
    	    (XtCallbackProc) rangeCancelCB, (void *) form);
    XmStringFree(xmstr);

    XtManageChild(form);
}

static void rangeOkCB(Widget w, Widget x, caddr_t call_data)
{
    /* This function will be called after the user has entered the 
       maximum and minimum values from the setRangeCB. This
       function first calls CurvesGetVisibleRange to get the current 
       range. If the user leaves one field blank, the field will not  
       be affected. If the user changed some fields, this function 
       converts the text from char to double and passes the value 
       to the function CurvesSetVisibleRange; */
    
    char *t;   
    double tempDouble, minXLim, maxXLim, minYLim, maxYLim;
    char *str, *ptr;
    
    /* Find the initial visible range to supply default values */
    CurvesGetVisibleRange(CurveWidget, &minXLim, &minYLim, &maxXLim,
    	    &maxYLim);
    
    /* Get text from the 4 widgets and convert string from char to double. */
    str = XmTextGetString(RangeText[0]);
    tempDouble = strtod(str, &ptr);
    if (ptr != str)
        minXLim = tempDouble;
    XtFree(str);
    
    str = XmTextGetString(RangeText[1]);
    tempDouble = strtod(str, &ptr); 
    if (ptr != str)
        minYLim = tempDouble;
    XtFree(str);
    
    str = XmTextGetString(RangeText[2]);
    tempDouble = strtod(str, &ptr);
    if (ptr != str)
        maxXLim = tempDouble; 
    XtFree(str);
    
    str = XmTextGetString(RangeText[3]);
    tempDouble = strtod(str, &ptr);
    if (ptr != str)
        maxYLim = tempDouble;   
    XtFree(str);
   
    if (minXLim >= maxXLim || minYLim >= maxYLim) 
	createWarningMessage(x, "Can't set visible range.\nCheck values");
    else { 
	XtDestroyWidget(XtParent(x));
        CurvesSetVisibleRange(CurveWidget, minXLim, minYLim, maxXLim,
        	maxYLim); 
    }
}

static void rangeCancelCB(Widget w, Widget x, caddr_t call_data)
{
    XtDestroyWidget(XtParent(x));
}

static void createWarningMessage(Widget parent, char *text)
{
    /* This function creates a warning dialog that displays 
       the text from parameter "text".  The dialogs created here
       are never destroyed, so don't use this in a real program */
        
    Widget message;
    Arg wargs[5];
    XmString xmstr, xmWarning;
    int n;
    
    /* Create a warning dialog widget to display a message. */
    n = 0;
    xmstr = XmStringCreateLtoR(text, XmSTRING_DEFAULT_CHARSET);
    xmWarning = XmStringCreateSimple("Warning");
    XtSetArg(wargs[n], XmNmessageString, xmstr);    n++;
    XtSetArg(wargs[n], XmNdialogTitle, xmWarning);   n++; 
    message = XmCreateWarningDialog(parent, "message", wargs, n);
    XmStringFree(xmstr);
    XmStringFree(xmWarning);
    
    /* Unmanage the Cancel and Help widgets, since we won't need them. */ 
    XtUnmanageChild(XmMessageBoxGetChild (message, XmDIALOG_CANCEL_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild (message, XmDIALOG_HELP_BUTTON));
    
    /* Manage and pop-up the dialog */
    XtManageChild(message);
}    

static void getRangeCB(Widget w, caddr_t client_data, caddr_t call_data)
{
   /* This function creates a message dialog that displays the coordinates
       that are returned from CurvesGetVisibleRange. */
       
    double minXLim, maxXLim, minYLim, maxYLim;
    Widget infoBox;
    XmString xmstr1, xmstr2;
    char message[100];
    Arg wargs[5];
    int n;
   
    CurvesGetVisibleRange(CurveWidget, &minXLim, &minYLim, &maxXLim,
    	    &maxYLim);           
   
    /* Create the string for display. */
    sprintf(message, "min X = %g, min Y = %g, max X = %g, max Y = %g",
            minXLim, minYLim, maxXLim, maxYLim);
    
    /* Create a messaage dialog to display coordinates. */
    n = 0;
    xmstr1 = XmStringCreateSimple("Visible Range");
    xmstr2 = XmStringCreateLtoR(message, XmSTRING_DEFAULT_CHARSET); 
    XtSetArg(wargs[n], XmNmessageString, xmstr2 ); n++;
    XtSetArg(wargs[n], XmNdialogTitle, xmstr1); n++; 
    infoBox = XmCreateMessageDialog(w, "infoBox", wargs, n);
    XmStringFree(xmstr1);  
    XmStringFree(xmstr2);   
      
    /* Unmanage the Cancel and Help widgets, since we won't need them. */ 
    XtUnmanageChild(XmMessageBoxGetChild (infoBox, XmDIALOG_CANCEL_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild (infoBox, XmDIALOG_HELP_BUTTON));
    
    XtManageChild(infoBox);
}

static void scaleVariablesCB(Widget w, caddr_t client_data, caddr_t call_data)
{
    /* This function calls CurvesScaleVariables, which allows the
       user to change the variable scaling. */
       
    CurvesScaleVariables(CurveWidget);
}

static void setMarkLineStyleCB(Widget w, caddr_t client_d, caddr_t call_data)
{
    /* This function calls CurvesSetMarkLineStyle, which allows the
       user to mark the critical points (the point where the curve changes 
       direction) with different styles of markers and also change the
       style of the line in each curve. */
       
    CurvesSetMarkLineStyle(CurveWidget);
}

static void showLegendCB(Widget w, caddr_t client_d, caddr_t call_d)
{
    /* This function calls CurvesShowLegend with parameter to show
       or hide the legend at the bottom of the plot */
        
    CurvesShowLegend(CurveWidget, XmToggleButtonGetState(w)); 
}

static void timeSeriesCB(Widget w, XtPointer client_d, caddr_t call_d)
{
    /* This function switches the widget back an forth between showing
       a time-series plot and an X-Y plot. */
       
    createCurves(XmToggleButtonGetState(TimeSeriesToggle),
    	    XmToggleButtonGetState(ErrorBarsToggle));
}

static void errorBarsCB(Widget w, XtPointer client_d, caddr_t call_d)
{
    /* This function switches the widget back an forth between showing
       a time-series plot and an X-Y plot. */
       
    createCurves(XmToggleButtonGetState(TimeSeriesToggle),
    	    XmToggleButtonGetState(ErrorBarsToggle));
}

static void createCurves(Boolean isTimeSeries, Boolean showErrorBars)
{
    static float data1[NUM_POINTS] = {1, 5, 12, 6, 18, 4, 6, 9, 14, 13};
    static float data2[NUM_POINTS] = {8, 6, 3, 2, 0, 5, 4, 2, 3, 6};
    static float data3[NUM_POINTS] = {15, 4, 6, 8 ,12, 9, 4, 8, 25, 2};
    static float tErr2data[NUM_POINTS] = {16, 5, 7, 9, 13, 10, 5, 9, 26, 3};
    static float bErr2data[NUM_POINTS] = {14, 3, 5, 7, 11, 8, 3, 7, 24, 1};
    static CurveStruct curves[5] = {
    	    {"Curve1", NUM_POINTS, SQUARE_MARK, PLAIN_LINE,
    	    	    CURVE_NO_OPTIONS, data1},
    	    {"Curve2", NUM_POINTS, BLANK_MARK,
    	    	    LINE_2, CURVE_NO_OPTIONS, data2},
    	    {"Curve3", NUM_POINTS, SOLID_CIRCLE_MARK,
    	    	    LINE_3, CURVE_NO_OPTIONS, data3},
    	    {"", NUM_POINTS, 0, 0, CURVE_TOP_ERROR, tErr2data},
    	    {"", NUM_POINTS, 0, 0, CURVE_BOTTOM_ERROR, bErr2data}
    };

    CurvesSetContents(CurveWidget, curves, showErrorBars ? 5 : 3,
    	    CURVES_RESCALE, isTimeSeries);
}


static void zoomCB(Widget w, XtPointer data, caddr_t call_data)
{
    /* This one callback is used for all 3 zoom pushbuttons:
       Zoom In, Zoom Out, and Reset Zoom. */
       
    switch ((int)data) {
        case 1 : CurvesZoomIn(CurveWidget); 
                 break;
        case 2 : CurvesZoomOut(CurveWidget);
                 break;
        case 3 : CurvesResetZoom(CurveWidget); 
                 break;         
    } 
}

static void printCB(Widget w, caddr_t client_d, caddr_t call_d)
{
    /* This routine prints a PostScript version of the plot.
       CurvesPrintContents writes a post script file that is passed to
       the utility routine PrintFile, which presents a dialog and
       queues the file for printing. */
       
    static char fileName[] = "temp.ps";
	
    CurvesPrintContents(CurveWidget, fileName); 
#ifdef VMS
    PrintFile(w, fileName, fileName, True);
#else
    PrintFile(w, fileName, fileName);
    remove(fileName);
#endif
}    
  
static void exitCB(Widget w, caddr_t client_data, caddr_t call_data)
{ 
    exit(0);
}
