/*
 * Java-Gnome Bindings Library
 * 
 * Copyright 1998-2002 the Java-Gnome Team, all rights reserved.
 * 
 * The Java-Gnome Team Members: Jean Van Wyk <jeanvanwyk@iname.com> Jeffrey S.
 * Morgan <jeffrey.morgan@bristolwest.com> Dan Bornstein <danfuzz@milk.com>
 * 
 * The Java-Gnome bindings library is free software distributed under the terms
 * of the GNU Library General Public License version 2.
 */
package org.gnu.gconf;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

import org.gnu.glib.GObject;

/**
 * 
 */
public class ConfClient extends GObject {
	
	private class NotifyData {
		public String namespace;
		public Vector listeners;
	}
	
	private class NotifyClient {
		public int id;
		public ConfClientListener lis;
	}
	
	/**
	 * Listeners for handling notify events
	 */
	private Hashtable listeners = null;
	
	private ConfClient(int handle) {
		this.handle = handle;
	}
	
	/**
	 * Method to get the default ConfClient
	 * @return The default ConfClient.
	 */
	public static ConfClient getInstance() {
		return new ConfClient(gconf_client_get_default());
	}
	
	/**
	 * 
	 * @param dir
	 * @param type
	 * @throws ConfException
	 */
	public void addDirectory(String dir, ConfClientPreloadType type) throws ConfException {
		int err = -1;
		gconf_client_add_dir(handle, dir, type.getValue(), new int[] {err});
		checkError(err);
	}

	/**
	 * 
	 * @param dir
	 * @throws ConfException
	 */
	public void removeDirectory(String dir) throws ConfException {
		int err = -1;
		gconf_client_remove_dir(handle, dir, new int[] {err});
		checkError(err);
	}

	

	/**
	 * Register an object to handle notify events.
	 * @see org.gnu.gconf.ConfClientListener
	 */
	public void addListener(ConfClientListener listener, String nameSpace)  throws ConfException {
		int err = -1;
		int id = gconf_client_notify_add(handle, nameSpace, new int[] {err});
		checkError(err);
		
		if (null == listeners)
				listeners = new Hashtable();
		NotifyData nd = (NotifyData)listeners.get(nameSpace);

		if (null == nd) {
			nd = new NotifyData();
			nd.namespace = nameSpace;
			nd.listeners = new Vector();
			listeners.put(nameSpace, nd);
		}
		NotifyClient nc = new NotifyClient();
		nc.id = id;
		nc.lis = listener;
		nd.listeners.addElement(nc);
	}

	/**
	 * Removes a listener
	 * @see #addListener(ConfClientListener)
	 */
	public void removeListener(ConfClientListener listener, String nameSpace) {
		int id = -1;
		
		// get the object from the hashtable.
		if (null == listeners)
			return;
		NotifyData nd = (NotifyData)listeners.get(nameSpace);
		if (null == nd)
			return;

		for (int i = 0; i < nd.listeners.size(); i++) {
			NotifyClient nc = (NotifyClient)nd.listeners.elementAt(i);
			if (nc.lis == listener) {
				id = nc.id;
				nd.listeners.remove(nc.lis);
				break;
			}
		}
		if (-1 == id)
			return;

		gconf_client_notify_remove(handle, id);
	}

	/**
	 * If you know you're done reading values for a while you can blow
	 * away the cache.  Note that this nullifies the effect of any preloading
	 * you may have done.  However it frees some memory.
	 */
	public void clearCache() {
		gconf_client_clear_cache(handle);
	}

	/**
	 * Preload a directory.  The directory must have been added already.  This
	 * is only useful as an optimization if you clear the cache, then later want to
	 * do a lot a reads again.
	 * @param directory The directory to load
	 * @param type How to preload the directory.
	 * @throws ConfException
	 */
	public void preload(String directory, ConfClientPreloadType type) throws ConfException {
		int err = -1;
		gconf_client_preload(handle, directory, type.getValue(), new int[] {err});
		checkError(err);
	}

	public void set(String key, ConfValue value) throws ConfException {
		int err = -1;
		gconf_client_set(handle, key, value.getHandle(), new int[] {err});
		checkError(err);
	}
	
	public ConfValue get(String key) throws ConfException {
		int err = -1;
		int hndl = gconf_client_get(handle, key, new int[] {err});
		checkError(err);
		return new ConfValue(hndl);
	}
	
	public ConfEntry getEntry(String key, String locale, boolean useSchemaDefault) throws ConfException {
		int err = -1;
		int hndl = gconf_client_get_entry(handle, key, locale, useSchemaDefault, new int[] {err});
		checkError(err);
		return new ConfEntry(hndl);
	}
	
	public ConfValue getDefaultFromSchema(String key) throws ConfException {
		int err = -1;
		int hndl = gconf_client_get_default_from_schema(handle, key, new int[] {err});
		checkError(err);
		return new ConfValue(hndl);
	}
	
	public boolean unset(String key) throws ConfException {
		int err = -1;
		boolean ret = gconf_client_unset(handle, key, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public List getAllEntries(String dir) throws ConfException {
		int err = -1;
		int [] array = gconf_client_all_entries(handle, dir, new int[] {err});
		checkError(err);
		if (array == null)
			return null;
		List ret = new ArrayList();
		for (int i = 0; i < array.length; i++) {
			ret.add(new ConfEntry(array[i]));
		}
		return ret;
	}
	
	public List getAllDirs(String dir) throws ConfException {
		int err = -1;
		String [] array = gconf_client_all_dirs(handle, dir, new int[] {err});
		checkError(err);
		if (array == null)
			return null;
		List ret = new ArrayList();
		for (int i = 0; i < array.length; i++) {
			ret.add(array[i]);
		}
		return ret;
	}
	
	public void suggestSync() throws ConfException {
		int err = -1;
		gconf_client_suggest_sync(handle, new int[] {err});
		checkError(err);
	}
	
	public boolean dirExists(String dir) throws ConfException {
		int err = -1;
		boolean ret = gconf_client_dir_exists(handle, dir, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public double getDouble(String key) throws ConfException {
		int err = -1;
		double ret = gconf_client_get_float(handle, key, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public int getInt(String key) throws ConfException {
		int err = -1;
		int ret = gconf_client_get_int(handle, key, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public String getString(String key) throws ConfException {
		int err = -1;
		String ret = gconf_client_get_string(handle, key, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public boolean getBoolean(String key) throws ConfException {
		int err = -1;
		boolean ret = gconf_client_get_bool(handle, key, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public ConfSchema getSchema(String key) throws ConfException {
		int err = -1;
		int ret = gconf_client_get_schema(handle, key, new int[] {err});
		checkError(err);
		return new ConfSchema(ret);
	}
	
	public Object[] getList(String key, ConfValueType listType) throws ConfException {
		int err = -1;
		Object [] list = gconf_client_get_list(handle, key, listType.getValue(), new int[] {err});
		checkError(err);
		return list;
	}
	
	public boolean setDouble(String key, double value) throws ConfException {
		int err = -1;
		boolean ret = gconf_client_set_float(handle, key, value, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public boolean setInt(String key, int value) throws ConfException {
		int err = -1;
		boolean ret = gconf_client_set_int(handle, key, value, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public boolean setString(String key, String value) throws ConfException {
		int err = -1;
		boolean ret = gconf_client_set_string(handle, key, value, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public boolean setBoolean(String key, boolean value) throws ConfException {
		int err = -1;
		boolean ret = gconf_client_set_bool(handle, key, value, new int[] {err});
		checkError(err);
		return ret;
	}
	
	public boolean setSchema(String key, ConfSchema value) throws ConfException {
		int err = -1;
		boolean ret = gconf_client_set_schema(handle, key, value.getHandle(), new int[] {err});
		checkError(err);
		return ret;
	}
	
	public boolean setList(String key, ConfValueType listType, List value) throws ConfException {
		// MUSTDO
		return true;
	}
	
	protected void notifyCallback(int e, String nameSpace) {
		NotifyData nd = (NotifyData)listeners.get(nameSpace);
		if (null == nd)
			return;
		ConfEntry entry = new ConfEntry(e);
		Vector notifyListeners = nd.listeners;
		if (null != notifyListeners) {
			for (int i = 0; i < notifyListeners.size(); i++) {
				NotifyClient nc = (NotifyClient)notifyListeners.elementAt(i);
				nc.lis.clientNotify(entry);
			}
		}
	}
	
	private void checkError(int error) throws ConfException {
		if (-1 < error)
			throw new ConfException(error);
	}

	/****************************************
	 * BEGINNING OF NATIVE CODE
	 ****************************************/
	native static final protected int gconf_client_get_default();
	native static final protected int gconf_client_get_for_engine(int engine);
	native static final protected void gconf_client_add_dir(int client, String dir, int preloadType, int[] error);
	native static final protected void gconf_client_remove_dir(int client, String dir, int[] error);
	native final protected int gconf_client_notify_add(int client, String namespace, int[] error);
	native static protected void gconf_client_notify_remove(int client, int cnxn);
	native static final protected void gconf_client_clear_cache(int client);
	native static final protected void gconf_client_preload(int client, String dirname, int type, int[] error);
	native static final protected void gconf_client_set(int client, String key, int val, int[] error);
	native static final protected int gconf_client_get(int client, String key, int[] error);
	native static final protected int gconf_client_get_without_default(int client, String key, int[] error);
	native static final protected int gconf_client_get_entry(int client, String key, String locale, boolean useSchemaDefault, int[] error);
	native static final protected int gconf_client_get_default_from_schema(int client, String key, int[] error);
	native static final protected boolean gconf_client_unset(int client, String key, int[] error);
	native static final protected int[] gconf_client_all_entries(int client, String dir, int[] error);
	native static final protected String[] gconf_client_all_dirs(int client, String dir, int[] error);
	native static final protected void gconf_client_suggest_sync(int client, int[] error);
	native static final protected boolean gconf_client_dir_exists(int client, String dir, int[] error);
	native static final protected double gconf_client_get_float(int client, String key, int[] error);
	native static final protected int gconf_client_get_int(int client, String key, int[] error);
	native static final protected String gconf_client_get_string(int client, String key, int[] error);
	native static final protected boolean gconf_client_get_bool(int client, String key, int[] error);
	native static final protected int gconf_client_get_schema(int client, String key, int[] error);
	native static final protected Object[] gconf_client_get_list(int client, String key, int listType, int[] error);
	native static final protected boolean gconf_client_get_pair(int client, String key, int carType, int cdrType, int[] car, int[] cdr, int[] error);
	native static final protected boolean gconf_client_set_float(int client, String key, double val, int[] error);
	native static final protected boolean gconf_client_set_int(int client, String key, int val, int[] error);
	native static final protected boolean gconf_client_set_string(int client, String key, String val, int[] error);
	native static final protected boolean gconf_client_set_bool(int client, String key, boolean val, int[] error);
	native static final protected boolean gconf_client_set_schema(int client, String key, int val, int[] error);
	//native static final protected boolean gconf_client_set_list(int client, String key, int listType, int[] list, int[] error);
	native static final protected boolean gconf_client_set_pair(int client, String key, int carType, int cdrType, int car, int cdr, int[] error);
	native static final protected void gconf_client_value_changed(int client, String key, int value);
	/****************************************
	 * END OF NATIVE CODE
	 ****************************************/
}
